/*
   Author: Don Capps (capps@iozone.org)
             7417 Crenshaw
            Plano, TX 75025

   The MIT License (MIT)

   Copyright (c) 2015 Don Capps

   Permission is hereby granted, free of charge, to any person obtaining a copy
   of this software and associated documentation files (the "Software"), to deal
   in the Software without restriction, including without limitation the rights
   to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
   copies of the Software, and to permit persons to whom the Software is
   furnished to do so, subject to the following conditions:

   The above copyright notice and this permission notice shall be included in
   all copies or substantial portions of the Software.

   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
   IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
   AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
   LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
   OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
   THE SOFTWARE.

*/

/* 
 * Iozone's personal Enigma_on_steroids encrypt/decrypt routines.
 * (This is a simple flexible symetric encryption algorithm)
 *
 * Author: Don Capps
 * Location: Iozone.org
 * Date: 3/1/2015
 *
 * MT = Mersenne Twister.
 * P & M = Park and Miller Linear Congruential random number generator.
 *
 * This is loosly based on the Enigma machine. Enigma, plus variable number of
 * rotors, and each rotor has 256 characters. The rotors are initialized from
 * a key that the user provides.  This key is used to seed two random number 
 * generators. One that shuffles the characters on the rotors, and another
 * that is used to XOR with input characters.  Each time a character is 
 * processed it is Xor'd with a changing random value (MT), and a random 
 * rotor will be re-shuffled (P & M), using either a per character partial 
 * shuffle (P & M), or a longer shuffle (P & M) at a user specified re-shuffle
 * frequency. 
 * 
 * The layers of cipher are:
 *   Xor with input character with random sequence. ( 64 bit Mersenne Twister )
 *   Substitution shift across N user defined rotors. ( Rotors randomized 
 *							with P & M )
 *   Per character partial shuffle of random rotor. ( P & M )
 *   More exhaustive shuffle of random rotor at user defined freq. (P & M)
 *
 * There are "num_rotors" rotors.  Each contains a randomized set of values
 * from 0 to 255.   Each letter is fed into the series of rotors, with
 * the input to the next rotor being the output from the previous rotor.
 *
 *  
 * Rotor # 0                         Rotor #1   ---->
 *
 * Index       _____Output____>                          __Output__>
 *             ^               |                         ^          |
 *             |               |                         |          |
 * 0   1   2   3   4   5       |     0   1   2   3   4   5          |
 * -   -   -   -   -   -       |     -   -   -   -   -   -          |
 * Input ----> 65              |__Input________________> 3          |___> 5
 *
 * The letter 'A' is decimal 65.  The first rotor converts this to 
 * the index value of 3.  The second rotor converts the input value of 3 into
 * the index of 5.  This continues through all of the randomized rotors.
 *
 * Note: The total number of rotors is configurable. (See num_rotors)
 * Currently there are 7 default rotors, each of which contains randomized
 * values.  The seed for the randomization, the number of rotors, and
 * the re-shuffle frequency are given by the user as parameters to the 
 * _start_encryption() routine. 
 *
 * ------------------------------------------------------------------------
 * Functions:
 *
 *    _start_encryption( key, num_rotors, reshuffle_freq)
 *      This starts the entire encryption mechanism;
 *
 *    _end_encryption(void)
 *       This frees all memory associated with encyption machine.
 *
 *    ret = _crypt_char( char )
 *       This encrypts a character, and returns the encrypted version.
 *
 *    ret = _decrypt_char( char )
 *       This decrypts a character, and returns the un-encrypted version.
 *
 *    ret = _crypt_string( unsigned char *dest, unsigned char *src, int count )
 *       This encrypts a string, and returns pointer to the destination 
 *
 *    ret = _decrypt_string(unsigned char *dest, unsigned char *src, int count )
 *       This decrypts a string, and returns pointer to the destination 
 *   --------------------------------------------------------------------
 *   Internal mechanisms 
 *   --------------------------------------------------------------------
 *    unsigned char _rotor_forward(int which, unsigned char input);
 *      Routine that takes which rotor, and initial character and finds
 *      the output character from this rotor, given this input character.
 *
 *    unsigned char _rotor_backward(int, unsigned char);
 *      Routine that takes which rotor, and initial character and finds
 *      the output character from the previous rotor.
 *
 *    void _sync_rotor(void)
 *      Routine that picks a rotor (based on the user's key) and randomises
 *      the contents of this rotor. It also advances a secondary key value as
 *      each character is processed. Thus some rotors are re-randomized 
 *      as characters of input are processed. (more than simply advanced
 *      as in the original Enigma machine)
 *
 *    void shuffle_two( struct rotor * );
 *      Routine that does a shuffle of two of the characterr in a rotor
 *      
 *    void _alloc_rotors(int);
 *      Allocates the user specified number of rotors.
 *
 *    struct rotor * _init_rotor( int which)
 *      Initializes the rotors with values, and shuffles each rotor's contents.
 *      Returns a pointer to the rotor.
 *
 * Legal note:  
 * This is an encryption mechanism and may come under some form of 
 * distribution restrictions.  See:
 * http://en.wikipedia.org/wiki/Export_of_cryptography_from_the_United_States	
 */

#include <stdio.h>
#include <stdlib.h>
#include <fcntl.h>
#include <string.h>

#define ROTORLIBVERSION "Rotor_lib $Revision: 1.14 $"

char rotor_lib_version[] = ROTORLIBVERSION;
static int rotor_seedi = 2231;

/* Default number of characters on each rotor */
#define NUM_POS 256

/* Distance, in characters, between re-shuffles */
#define COG 13
int cog = COG;

/* Default number of rotors */
#define NUM_ROT 7
int num_rotors = NUM_ROT; 	/* Defaults */

struct rotor {
	unsigned char ch_array[NUM_POS];   /* The character */
	unsigned char ch_iarray[NUM_POS];  /* index of character */
}global_rotor;

int global_start_enc=0;

/* Prototypes */
/* Exported externally */
void _start_encryption(unsigned long long,int,int);
void _end_encryption(void);
unsigned char _crypt_char( unsigned char );
unsigned char _decrypt_char(unsigned char);
unsigned char * _crypt_string( unsigned char *, unsigned char *, int );
unsigned char * _decrypt_string(unsigned char *, unsigned char *, int);

double fabs(double);
double log(double);

void   _RandomInitialise(int,int);
double _RandomUniform(void);
double _RandomGaussian(double,double);
int    _RandomInt(int,int);
double _RandomDouble(double,double);
/* Internal only */
static struct rotor *_init_rotor( int );
static void _allocate_all_rotors(int);
static void _dealloc_rotors(int);
static unsigned char _rotor_forward(int, unsigned char);
static unsigned char _rotor_backward(int, unsigned char);
static void _sync_rotor(void);
static void _alloc_rotors(int);
static double _rotor_ran(void);
static int _rotor_rand(void);
static void _rotor_srand(int);
static void shuffle_two( struct rotor * );
void rotor_init_genrand64(unsigned long long );
unsigned long long rotor_genrand64_int64(void);
#if defined(_USED_)
extern static void rotor_init_by_array64(unsigned long long *, unsigned long long );
long long rotor_genrand64_int63(void); 
double rotor_genrand64_real1(void);
double rotor_genrand64_real2(void);
double rotor_genrand64_real3(void);
#endif

struct rotor **r_array; /* Array of all of the rotors */
unsigned long long key;	/* The seed for the srand48(); */
int syncp=1;	/* Advances for each input character */

#ifdef TEST
main()
{
	unsigned char in[50];
	unsigned char out[50];
	unsigned char outf[50];
	unsigned char *dest;
	int len;

	bzero(in,5);
	bzero(out,5);
	bzero(outf,5);

	strcpy(in,"Hello world\n");
	len = strlen(in);

	printf("Started with %s\n",in);

	_start_encryption((unsigned long long)12,8, 5);
	dest = _crypt_string(out, in, len);
	_end_encryption();
	printf("Encrypted: %s\n",dest);

	_start_encryption((unsigned long long)12,8, 5);
	dest = _decrypt_string(outf,out,len);
	_end_encryption();

	printf("Decrypted: %s\n",dest);
}
#endif
	
/* 
 * Re-shuffle the contents of a randomly selected rotor
 */
static void
_sync_rotor()
{
	int rd;
	int which_rotor,how_much;
	unsigned char rand1, rand2,s1,s2,tmp;
	struct rotor *rotor_ptr;
	

	/* Move the seed for each input character */
	syncp++;
	
	if(syncp % cog != 0)
	{	
		/* Minor shuffle */
		which_rotor  = ((_rotor_rand()>>2) % num_rotors);
		rotor_ptr = r_array[which_rotor];
		shuffle_two(rotor_ptr);
		return;
	}
	
	/* Prepare for the big shuffle */

	/* Pick a rotor to re-spin */
	which_rotor  = ((_rotor_rand()>>2) % num_rotors);

	/* Shuffle an average of 1/2 of the total number of positions */
	how_much  = ((_rotor_rand()>>3) % NUM_POS);

	rotor_ptr = r_array[which_rotor];

	/* The big shuffle */
	for(rd = 0; rd < (how_much); rd++)
	{
	   	   rand1 = (unsigned char) ((_rotor_rand()>>2) % NUM_POS);
	   	   rand2 = (unsigned char )((_rotor_rand()>>2) % NUM_POS);
	   	   s1 = rotor_ptr->ch_array[(unsigned char)rand1];
	   	   s2 = rotor_ptr->ch_array[(unsigned char)rand2];
	   	   rotor_ptr->ch_array[(unsigned char)rand1] = s2;
	   	   rotor_ptr->ch_array[(unsigned char)rand2] = s1;
		   /* Swap the reverse lookup too */
		   tmp = rotor_ptr->ch_iarray[rotor_ptr->ch_array[rand2]];
		   rotor_ptr->ch_iarray[rotor_ptr->ch_array[rand2]] = 
			   rotor_ptr->ch_iarray[rotor_ptr->ch_array[rand1]];
		   rotor_ptr->ch_iarray[rotor_ptr->ch_array[rand1]] = tmp;
	}
}

/*
 * A minor shuffle. This is called more frequently, and only does
 * a little shuffle of a randomly chosen rotor. 
 */
static void
shuffle_two( struct rotor *rotor_ptr )
{
	unsigned char rand1, rand2,s1,s2,tmp;

	/* Shuffle two locations */
   	rand1 = (unsigned char) ((_rotor_rand()>>2) % NUM_POS);
	rand2 = (unsigned char )((_rotor_rand()>>2) % NUM_POS);
	s1 = rotor_ptr->ch_array[(unsigned char)rand1];
	s2 = rotor_ptr->ch_array[(unsigned char)rand2];
	rotor_ptr->ch_array[(unsigned char)rand1] = s2;
	rotor_ptr->ch_array[(unsigned char)rand2] = s1;

	/* Swap the reverse lookup too */
	tmp = rotor_ptr->ch_iarray[rotor_ptr->ch_array[rand2]];
	rotor_ptr->ch_iarray[rotor_ptr->ch_array[rand2]] = 
		rotor_ptr->ch_iarray[rotor_ptr->ch_array[rand1]];
	rotor_ptr->ch_iarray[rotor_ptr->ch_array[rand1]] = tmp;
}

static struct rotor *
_init_rotor( int which)
{
	int i,rd,how_much;
	unsigned char rand1, rand2,s1,s2;
	struct rotor *rotor_ptr;
	rotor_ptr = (struct rotor *)malloc(sizeof(struct rotor));
	how_much = NUM_POS/2;

	for(i = 0; i < NUM_POS; i++)
	{
		global_rotor.ch_array[i] = i;
	}
	memcpy(rotor_ptr->ch_array,&global_rotor.ch_array,NUM_POS);

	/* Randomize the contents of the rotor */
	for(rd = 0; rd < (how_much); rd++)
	{
	   rand1 = (unsigned char) (( _rotor_rand()>>2) % NUM_POS);
	   rand2 = (unsigned char )(( _rotor_rand()>>2) % NUM_POS);
	   s1 = rotor_ptr->ch_array[(unsigned char)rand1];
	   s2 = rotor_ptr->ch_array[(unsigned char)rand2];
	   rotor_ptr->ch_array[(unsigned char)rand1] = s2;
	   rotor_ptr->ch_array[(unsigned char)rand2] = s1;

	}
	/* Create the inverted lookup */
	for(i = 0; i < NUM_POS; i++)
	{
		rotor_ptr->ch_iarray[rotor_ptr->ch_array[i]] = i;
	}
	return(rotor_ptr);
}
	
static unsigned char
_rotor_forward( int which_rotor, unsigned char input )
{
	struct rotor *rotor_ptr;
	unsigned char offset;
	rotor_ptr = r_array[which_rotor];
	/* Return index of this character */
	offset=rotor_ptr->ch_iarray[input];
	return(offset);
}

static unsigned char
_rotor_backward( int which_rotor, unsigned char offset )
{
	unsigned char current_char;
	struct rotor *rotor_ptr;
	rotor_ptr = r_array[which_rotor];
	current_char = rotor_ptr->ch_array[offset];
	return(current_char);
}

static void
_alloc_rotors(int num_rot)
{
       /* Array of all of the pointers to the rotors */
       r_array = (struct rotor **)malloc(sizeof(struct rotor *) * num_rotors);  
}

/*
 * Deallocate the rotors, and the memory for the pointers to the rotors.
 */

static void
_dealloc_rotors(int num_rot)
{
	int i;
	for(i=0;i<num_rot;i++)
	{
	   free(r_array[i]);
	}
	free(r_array);
	r_array=(struct rotor **)NULL;
}

/*
 * -------------------------  External Definitions  -------------------------
 */

/* 
 * The Park and Miller LC random number generator.
 */

/*
 * Seed the random number generator.
 */
static void
_rotor_srand(
    int seed)
{
    rotor_seedi = seed;
}

/*
 * Returns a random number.
 */
static int
_rotor_rand(void)
{
    (void) _rotor_ran();
    return(rotor_seedi);
}

/*
 * Get the current seed value 
 */
/*  Not defined until needed..... 
static int
_rotor_gseed(void)
{
    return(rotor_seedi);
}
*/

/*
 * Compute the next random number.
 */
static double
_rotor_ran(void)

/***************************************************************/
/* See "Random Number Generators: Good Ones Are Hard To Find", */
/*     Park & Miller, CACM 31#10 October 1988 pages 1192-1201. */
/***************************************************************/
/* THIS IMPLEMENTATION REQUIRES AT LEAST 32 BIT INTEGERS !     */
/***************************************************************/

#if defined(WIN32)
#define int32_t int
#endif

#define _ROTOR_A_MULTIPLIER  16807L
#define _ROTOR_M_MODULUS     2147483647L /* (2**31)-1 */
#define _ROTOR_Q_QUOTIENT    127773L     /* 2147483647 / 16807 */
#define _ROTOR_R_REMAINDER   2836L       /* 2147483647 % 16807 */
{
    int32_t	lo;
    int32_t	hi;
    int32_t	test;

    hi = rotor_seedi / _ROTOR_Q_QUOTIENT;
    lo = rotor_seedi % _ROTOR_Q_QUOTIENT;
    test = _ROTOR_A_MULTIPLIER * lo - _ROTOR_R_REMAINDER * hi;
    if (test > 0) {
	rotor_seedi = test;
    } else {
	rotor_seedi = test + _ROTOR_M_MODULUS;
    }
    return((float) rotor_seedi / _ROTOR_M_MODULUS);
}

static void
_allocate_all_rotors(int num_rotors)
{
	int rotors;
	/* Allocate rotors */
	_alloc_rotors(num_rotors);

	/* Initalize the rotors */
	for(rotors=0; rotors < num_rotors; rotors++)
	{
	  r_array[rotors] = _init_rotor( rotors );
	}
}

unsigned char
_crypt_char( unsigned char buffer )
{
	unsigned char val;
	int spin;
	unsigned char rnum;
#if defined (_OLD_)
       	rnum = ( ( _rotor_rand() >> 2) & 0xff);
       	val = buffer ^ rnum;
#else
       	rnum= ( ( rotor_genrand64_int64() >> 2) & 0xff);
       	val = buffer ^ rnum;
#endif
       	for(spin=0; spin < num_rotors; spin++)
	{
	          val = _rotor_forward( spin, val);
	}
	_sync_rotor();
	return(val);
}

unsigned char *
_crypt_string( unsigned char *dest, unsigned char *src, int count )
{
	int j;
	for(j=0;j<count;j++)
	  dest[j]=_crypt_char(src[j]); 
	return(dest);
}

unsigned char *
_decrypt_string( unsigned char *dest, unsigned char *src, int count )
{
	int j;
	for(j=0;j<count;j++)
	  dest[j]=_decrypt_char(src[j]); 
	return(dest);
}

unsigned char
_decrypt_char(unsigned char buffer)
{
	unsigned char nval;
	int spin;
	nval = buffer;
	for(spin=num_rotors-1; spin >=0; spin--)
	{
   		nval = _rotor_backward( spin, nval);
	}
#if defined(_OLD_)
	nval = nval ^ ((_rotor_rand()>>2) & 0xff);
#else
	nval = nval ^ ((rotor_genrand64_int64() >> 2 ) & 0xff);
#endif
	_sync_rotor();
	return(nval);
}

void
_start_encryption( unsigned long long pkey, int num_rot, int reshuffle_freq)
{
	if(global_start_enc==1)
		return;
	syncp = 1;
	key = pkey;
	cog = reshuffle_freq;
	num_rotors = num_rot;
	_rotor_srand( (int)(key + (syncp * 17)));
	rotor_init_genrand64((unsigned long long)(key + syncp *17 ));
	_allocate_all_rotors(num_rotors);
	global_start_enc=1;
}

void
_end_encryption(void)
{
	if(global_start_enc==0)
		return;
	_dealloc_rotors(num_rotors);
	global_start_enc=0;
}

/*
 * This random number generator is used to help secure the users passphrase.
 */

/*
   This Random Number Generator is based on the algorithm in a FORTRAN
   version published by George Marsaglia and Arif Zaman, Florida State
   University; ref.: see original comments below.
   At the fhw (Fachhochschule Wiesbaden, W.Germany), Dept. of Computer
   Science, we have written sources in further languages (C, Modula-2
   Turbo-Pascal(3.0, 5.0), Basic and Ada) to get exactly the same test
   results compared with the original FORTRAN version.
   April 1989
   Karl-L. Noell <NOELL@DWIFH1.BITNET>
      and  Helmut  Weber <WEBER@DWIFH1.BITNET>

   This random number generator originally appeared in "Toward a Universal
   Random Number Generator" by George Marsaglia and Arif Zaman.
   Florida State University Report: FSU-SCRI-87-50 (1987)
   It was later modified by F. James and published in "A Review of Pseudo-
   random Number Generators"
   THIS IS THE BEST KNOWN RANDOM NUMBER GENERATOR AVAILABLE.
   (However, a newly discovered technique can yield
   a period of 10^600. But that is still in the development stage.)
   It passes ALL of the tests for random number generators and has a period
   of 2^144, is completely portable (gives bit identical results on all
   machines with at least 24-bit mantissas in the floating point
   representation).
   The algorithm is a combination of a Fibonacci sequence (with lags of 97
   and 33, and operation "subtraction plus one, modulo one") and an
   "arithmetic sequence" (using subtraction).

   Use IJ = 1802 & KL = 9373 to test the random number generator. The
   subroutine RANMAR should be used to generate 20000 random numbers.
   Then display the next six random numbers generated multiplied by 4096*4096
   If the random number generator is working properly, the random numbers
   should be:
           6533892.0  14220222.0  7275067.0
           6172232.0  8354498.0   10633180.0
*/

/* Globals */
#define _FALSE 0
#define _TRUE 1

double u[97],c,cd,cm;
int i97,j97;
int _test = _FALSE;

/*
   This is the initialization routine for the random number generator.
   NOTE: The seed variables can have values between:    0 <= IJ <= 31328
                                                        0 <= KL <= 30081
   The random number sequences created by these two seeds are of sufficient
   length to complete an entire calculation with. For example, if sveral
   different groups are working on different parts of the same calculation,
   each group could be assigned its own IJ seed. This would leave each group
   with 30000 choices for the second seed. That is to say, this random
   number generator can create 900 million different subsequences -- with
   each subsequence having a length of approximately 10^30.
*/
void _RandomInitialise(int ij,int kl)
{
   double s,t;
   int ii,i,j,k,l,jj,m;

   /*
      Handle the seed range errors
         First random number seed must be between 0 and 31328
         Second seed must have a value between 0 and 30081
   */
   if (ij < 0 || ij > 31328 || kl < 0 || kl > 30081) {
		ij = 1802;
		kl = 9373;
   }

   i = (ij / 177) % 177 + 2;
   j = (ij % 177)       + 2;
   k = (kl / 169) % 178 + 1;
   l = (kl % 169);

   for (ii=0; ii<97; ii++) {
      s = 0.0;
      t = 0.5;
      for (jj=0; jj<24; jj++) {
         m = (((i * j) % 179) * k) % 179;
         i = j;
         j = k;
         k = m;
         l = (53 * l + 1) % 169;
         if (((l * m % 64)) >= 32)
            s += t;
         t *= 0.5;
      }
      u[ii] = s;
   }

   c    = 362436.0 / 16777216.0;
   cd   = 7654321.0 / 16777216.0;
   cm   = 16777213.0 / 16777216.0;
   i97  = 97;
   j97  = 33;
   _test = _TRUE;
}

/* 
   This is the random number generator proposed by George Marsaglia in
   Florida State University Report: FSU-SCRI-87-50
*/
double _RandomUniform(void)
{
   double uni;

   /* Make sure the initialisation routine has been called */
   if (!_test) 
   	_RandomInitialise(1802,9373);

   uni = u[i97-1] - u[j97-1];
   if (uni <= 0.0)
      uni++;
   u[i97-1] = uni;
   i97--;
   if (i97 == 0)
      i97 = 97;
   j97--;
   if (j97 == 0)
      j97 = 97;
   c -= cd;
   if (c < 0.0)
      c += cm;
   uni -= c;
   if (uni < 0.0)
      uni++;

   return(uni);
}

/*
  ALGORITHM 712, COLLECTED ALGORITHMS FROM ACM.
  THIS WORK PUBLISHED IN TRANSACTIONS ON MATHEMATICAL SOFTWARE,
  VOL. 18, NO. 4, DECEMBER, 1992, PP. 434-435.
  The function returns a normally distributed pseudo-random number
  with a given mean and standard devaiation.  Calls are made to a
  function subprogram which must return independent random
  numbers uniform in the interval (0,1).
  The algorithm uses the ratio of uniforms method of A.J. Kinderman
  and J.F. Monahan augmented with quadratic bounding curves.
*/
double _RandomGaussian(double mean,double stddev)
{
   double  q,u,v,x,y;

	/*  
		Generate P = (u,v) uniform in rect. enclosing acceptance region 
      Make sure that any random numbers <= 0 are rejected, since
      gaussian() requires uniforms > 0, but RandomUniform() delivers >= 0.
	*/
   do {
      u = _RandomUniform();
      v = _RandomUniform();
   	if (u <= 0.0 || v <= 0.0) {
       	u = 1.0;
       	v = 1.0;
   	}
      v = 1.7156 * (v - 0.5);

      /*  Evaluate the quadratic form */
      x = u - 0.449871;
   	y = fabs(v) + 0.386595;
      q = x * x + y * (0.19600 * y - 0.25472 * x);

      /* Accept P if inside inner ellipse */
      if (q < 0.27597)
			break;

      /*  Reject P if outside outer ellipse, or outside acceptance region */
    } while ((q > 0.27846) || (v * v > -4.0 * log(u) * u * u));

    /*  Return ratio of P's coordinates as the normal deviate */
    return (mean + stddev * v / u);
}

/*
   Return random integer within a range, lower -> upper INCLUSIVE
*/
int _RandomInt(int lower,int upper)
{
   return((int)(_RandomUniform() * (upper - lower + 1)) + lower);
}

/*
   Return random float within a range, lower -> upper
*/
double _RandomDouble(double lower,double upper)
{
   return((upper - lower) * _RandomUniform() + lower);
}

