/*
   Author: Don Capps (capps@iozone.org)
             7417 Crenshaw
            Plano, TX 75025

   The MIT License (MIT)

   Copyright (c) 2015 Don Capps

   Permission is hereby granted, free of charge, to any person obtaining a copy
   of this software and associated documentation files (the "Software"), to deal
   in the Software without restriction, including without limitation the rights
   to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
   copies of the Software, and to permit persons to whom the Software is
   furnished to do so, subject to the following conditions:

   The above copyright notice and this permission notice shall be included in
   all copies or substantial portions of the Software.

   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
   IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
   AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
   LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
   OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
   THE SOFTWARE.

*/

/* 
 * Iozone's personal enigma_on_steroids encrypt/decrypt routines.
 * (This is a symetric encryption algorithm)
 *
 * Author: Don Capps
 * Location: Iozone.org
 * Date: 3/1/2015
 *
 * This is loosly based on the Enigma machine. (Enigma plus variable number of
 * rotors, and each rotor has 256 characters. The rotors are initialized from
 * a key that the user provides.  This key is used to seed a random number 
 * generator, that then shuffles the characters on the rotors.  Also, each
 * time a character is processed the key is incremented, and a random rotor
 * will be re-shuffled)
 *
 *  rotor: [-e] [-d] [-k keyvalue | -p password] [-i infile] [-o outfile] 
 *         [-r num_rotors] [-c re-shuffle freq]
 *         
 *      -e Encrypt file
 *      -d Decrypt file
 *      -k ### Key value
 *      -p password
 *      -i filename. Input filename to process
 *      -I Input from stdin
 *      -o filename. Output filename to process
 *      -O Output to stdout
 *      -r ### Sets the number of rotors
 *      -c ### Sets the re-shuffle frequency
 * 
 * ------------------------------------------------------------------------
 *    Inside of rotor_lib.c are the routines that implement the cipher.
 * ------------------------------------------------------------------------
 *
 * There are "num_rotors" rotors.  Each contains a randomized set of values
 * from 0 to 255.   Each letter is fed into the series of rotors, with
 * the input to the next rotor being the output from the previous rotor.
 *
 *
 *
 * Rotor # 0                         Rotor #1   ---->
 *
 * Index       _____Output____>                          __Output__>
 *             ^               |                         ^          |
 *             |               |                         |          |
 * 0   1   2   3   4   5       |     0   1   2   3   4   5          |
 * -   -   -   -   -   -       |     -   -   -   -   -   -          |
 * Input ----> 65              |__Input________________> 3          |___> 5
 *
 * The letter 'A' is decimal 65.  The first rotor converts this to 
 * the index value of 3.  The second rotor converts the input value of 3 into
 * the index of 5.  This continues through all of the randomized rotors.
 *
 * Note: The total number of rotors is configurable (-r #). (See num_rotors)
 * Currently there are 7 default rotors, each of which contains randomized
 * values.  The seed for the randomization is given by the user
 * as a parameter to the encode and decode.
 *
 * External functions:
 *
 *    _start_encryption( key, num_rotors, re-shuffle_frequency)
 *      This starts the entire encryption mechanism;
 *
 *    _end_encryption( void )
 *       This frees all memory associated with encyption machine.
 *
 *    ret = _crypt_char( char )
 *       This encrypts a character, and returns the encrypted version.
 *
 *    ret = _decrypt_char( char )
 *       This decrypts a character, and returns the un-encrypted version.
 *
 *    ret = _crypt_string( unsigned char *dest, unsigned char *src, int count )
 *       This encrypts a string, and returns pointer to the destination 
 *
 *    ret = _decrypt_string(unsigned char *dest, unsigned char *src, int count )
 *       This decrypts a string, and returns pointer to the destination 
 */
#ifdef _WIN32
#include <Windows.h>
#endif
#ifdef _WIN32
#pragma warning(disable:4996)
#pragma warning(disable:4267)
#pragma warning(disable:4133)
#pragma warning(disable:4244)
#pragma warning(disable:4102)
#pragma warning(disable:4018)
#endif

#include <stdio.h>
#include <stdlib.h>
#include <fcntl.h>
#include <string.h>
#ifndef _WIN32
#include <unistd.h>
#endif
#define THISVERS "Rotor $Revision: 1.22 $"
char myversion[]=THISVERS;
extern int argc;
extern char **argv;
extern char *optarg;
extern char *optarg;
extern char rotor_lib_version[];
FILE *ifd,*ofd;
unsigned char *inbuf, *outbuf;
#define FBUFSIZE (32 * 1024)
/* Input flags */
int ifile,ofile,ikey,irot,ienc,idec,icog;

/* Defaults */
int mynum_rotors = 7;
int more_intense;
int cogs = 13;

/* Prototypes */
extern void _start_encryption(unsigned long long,int,int);
extern void _end_encryption(void);
extern unsigned char _crypt_char( unsigned char );
extern unsigned char _decrypt_char(unsigned char);
void usage (void);
void   _RandomInitialise(int,int);
extern int _RandomInt(int,int);

unsigned long long key = 421782LL;	
unsigned long long local_key = 421782LL;
unsigned char myphrase[256];
char infile[4096]; 	/* Input file */
char outfile[4096]; 	/* Output file */

/* The entry point */
int
main(int argc, char **argv)
{
	unsigned char val,nval,buffer;
	int cret,jj;


	/* 
	   Allocate some buffers so that we help fread & fwrite use
	   larger buffers for physical I/O. This will help fread and fwrite 
	   work more efficiently.
	*/
	inbuf = (unsigned char *)malloc(FBUFSIZE);
	outbuf = (unsigned char *)malloc(FBUFSIZE);

        while((cret = getopt(argc,argv,"hedv2i:o:k:r:c:p: ")) != EOF){
                switch(cret){
			/* Input file name */
		   case 'i': 
			strcpy(infile,optarg);
			ifile++;
			break;
			/* Output file name */
		   case 'o': strcpy(outfile,optarg);
			ofile++;
			break;
			/* Key used for seed */
		   case 'k':  sscanf(optarg,"%llu",&key);
			ikey++;
			break;
		   case 'p': strcpy((char *)myphrase,optarg);
			for(jj=0;jj<strlen((char *)myphrase);jj++)
			   local_key = myphrase[jj] * 17LL * local_key;
   			_RandomInitialise(1802,(int)key);
			key=(unsigned long long)local_key^_RandomInt(1,0x7FFFFFFF);
			ikey++;
			break;
			/* Sets number of rotors */
		   case 'r': mynum_rotors = (int) atoi(optarg);
			if( mynum_rotors < 1)
				mynum_rotors=1;
			irot++;
			break;
			/* We are encrypting */
		   case 'e': ienc++;
			break;
			/* We are decrypting */
		   case 'd': idec++;
			break;
		   case 'c': cogs = (int) atoi(optarg);
			if( cogs < 1)
				cogs=1;
			icog++;
			break;
			/* Help menu */
		   case 'h': 
			usage();
			exit(1);
			break;
		   case 'v': 
			printf("%s\n",myversion);
			printf("%s\n",rotor_lib_version);
			exit(0);
			break;
		   case '2': 
			more_intense=1;
			break;
		   default: exit(1);
			break;
		}
	}
	/* If using a password and the number 2 option */
	if((more_intense != 0) && (ikey != 0))
	{
		/* More rotors */
		if(!irot)
			mynum_rotors= (abs(((int)key)) % 251)+7;
		/* More shuffles */
		if(!icog)
			cogs = ((abs((int)key)) % 7) + 1;
	}
	/* Check to make sure we have all that we need */
	if( !(  ikey && (idec || ienc)) )
	{
		usage();
		exit(1);
	}
	/* Check to make sure we are doing one or the other */
	if((!ienc && !idec) || (ienc && idec))
	{
	   printf("Must choose to option to encrypt or decrypt\n");
	   exit(1);
	}
	/* Check to make sure we have both file names */

	if( ofile )
	{
	    printf("\n");
	    printf("\t--------------------------------------------\n");
	    printf("\t| Rotor-Enigma encryption, from Iozone.org |\n");
	    printf("\t|                   by                     |\n");
	    printf("\t|                Don Capps                 |\n");
	    printf("\t--------------------------------------------\n");
	}

	_start_encryption(key,mynum_rotors,cogs);

	/* If we are encrypting */
	if(ienc)
	{
#ifdef _WIN32
	    if(ifile)
	        ifd = fopen(infile,"rb");
	    else
	    {
	        ifd = stdin;
	  	_set_fmode(O_BINARY);
	    }

#else
	    if(ifile)
	        ifd = fopen(infile,"r");
	    else
	        ifd = stdin;
#endif
	   if( ifd == 0)
	   {
		printf("Unable to open file %s\n",infile);
		exit(1);
	   }
#ifdef _WIN32
	   if(ofile)
	       ofd = fopen(outfile,"wb");
	   else
	   {
	       ofd = stdout;
	  	_set_fmode(O_BINARY);
	   }
#else
	   if(ofile)
	       ofd = fopen(outfile,"w");
	   else
	       ofd = stdout;
#endif

	   /* 
	      Use the allocated buffers so that we help fread & fwrite use
	      larger buffers for physical I/O. This will help fread and fwrite 
	      work more efficiently.
	   */
           setvbuf(ifd, (char *)inbuf, _IOFBF , (FBUFSIZE));
           setvbuf(ofd, (char *)outbuf, _IOFBF , (FBUFSIZE));

   	   while( fread(&buffer,1,1,ifd) > 0)
	   {
	       val = _crypt_char(buffer);
	       fwrite(&val,1,1,ofd);
	   }
	   fflush(ofd);
	   fclose(ofd);
	   fclose(ifd);
	   if( ofile )
	       printf("\n");
	   _end_encryption();
	}

	/* If we are decrypting */
	if(idec)
	{
	   /* Now decode */
#ifdef _WIN32
           if(ifile)
	   {
	       ifd = fopen(infile,"rb");
	   }
           else
	   {
	        ifd = stdin;
	  	_set_fmode(O_BINARY);
	   }
#else
           if(ifile)
	       ifd = fopen(infile, "r");
           else
	       ifd = stdin;
#endif
	   if(ifd < 0)
	   {
		printf("Unable to open decode file \n");
		exit(1);
	   }
#ifdef _WIN32
	   if(ofile)
	       ofd = fopen(outfile,"wb");
	   else
	   {
	       ofd = stdout;
	  	_set_fmode(O_BINARY);
	   }
#else
	   if(ofile)
	       ofd = fopen(outfile, "w");
	   else
	       ofd = stdout;
#endif
	   if(ofd == 0)
	   {
		printf("Unable to open output file %s\n",outfile);
		exit(1);
	   }

	   /* 
	      Use the allocated buffers so that we help fread & fwrite use
	      larger buffers for physical I/O. This will help fread and fwrite 
	      work more efficiently.
	   */
           setvbuf(ifd, (char *)inbuf, _IOFBF , (FBUFSIZE));
           setvbuf(ofd, (char *)outbuf, _IOFBF , (FBUFSIZE));

	   val = 0;
   	   while( fread(&val,1,1,ifd) > 0)
	   {
		nval = _decrypt_char(val);
	        fwrite(&nval,1,1,ofd);
	   }
	   if( ofile )
	       printf("\n");
	   fflush(ofd);
	   fclose(ofd);
	   fclose(ifd);
	   _end_encryption();
	}
	return(0);
}

void
usage (void)
{
	printf("rotor: [-e] [-d] [-k keyvalue or -p password] [-i infile] [-o outfile]\n");
	printf("       [-r num_rotors] [-c reshuffle_freq ]\n");
	printf("       -----------------------------------------------------------\n");
	printf("        -e Encrypt file\n");
	printf("        -d Decrypt file\n");
	printf("        -k ### Key value\n");
	printf("        -p string.  Sets the key value based on a phrase.\n");
	printf("        -i filename. Input filename to process\n");
	printf("        -o filename. Output filename to process\n");
	printf("        -r ### Sets the number of rotors. (default 7)\n");
	printf("        -c ### Sets the re-shuffle frequency. (default 13)\n");
	printf("        -2 Increases the intensity of the encryption when used with -p \n");
}

