/* 
   A C-program for MT19937-64 (2004/9/29 version).
   Coded by Takuji Nishimura and Makoto Matsumoto.

   This is a 64-bit version of Mersenne Twister pseudorandom number
   generator.

   Before using, initialize the state by using rotor_init_genrand64(seed)  
   or init_by_array64(init_key, key_length).

   Copyright (C) 2004, Makoto Matsumoto and Takuji Nishimura,
   All rights reserved.                          

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions
   are met:

     1. Redistributions of source code must retain the above copyright
        notice, this list of conditions and the following disclaimer.

     2. Redistributions in binary form must reproduce the above copyright
        notice, this list of conditions and the following disclaimer in the
        documentation and/or other materials provided with the distribution.

     3. The names of its contributors may not be used to endorse or promote 
        products derived from this software without specific prior written 
        permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE COPYRIGHT OWNER OR
   CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
   EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
   PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
   PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
   LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
   NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
   SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

   References:
   T. Nishimura, ``Tables of 64-bit Mersenne Twisters''
     ACM Transactions on Modeling and 
     Computer Simulation 10. (2000) 348--357.
   M. Matsumoto and T. Nishimura,
     ``Mersenne Twister: a 623-dimensionally equidistributed
       uniform pseudorandom number generator''
     ACM Transactions on Modeling and 
     Computer Simulation 8. (Jan. 1998) 3--30.

   Any feedback is very welcome.
   http://www.math.hiroshima-u.ac.jp/~m-mat/MT/emt.html
   email: m-mat @ math.sci.hiroshima-u.ac.jp (remove spaces)
*/



#define ROTOR_NN 312
#define ROTOR_MM 156
#define ROTOR_MATRIX_A 0xB5026F5AA96619E9ULL
#define ROTOR_UM 0xFFFFFFFF80000000ULL /* Most significant 33 bits */
#define ROTOR_LM 0x7FFFFFFFULL /* Least significant 31 bits */


/* The array for the state vector */
static unsigned long long rotor_mt[ROTOR_NN]; 
/* rotor_mti==ROTOR_NN+1 means rotor_mt[ROTOR_NN] is not initialized */
static int rotor_mti=ROTOR_NN+1; 

/* initializes rotor_mt[ROTOR_NN] with a seed */
void rotor_init_genrand64(unsigned long long seed)
{
    rotor_mt[0] = seed;
    for (rotor_mti=1; rotor_mti<ROTOR_NN; rotor_mti++) 
        rotor_mt[rotor_mti] =  (6364136223846793005ULL * (rotor_mt[rotor_mti-1] ^ (rotor_mt[rotor_mti-1] >> 62)) + rotor_mti);
}

#if defined(_USED_)
/* initialize by an array with array-length */
/* init_key is the array for initializing keys */
/* key_length is its length */
void rotor_init_by_array64(unsigned long long init_key[],
		     unsigned long long key_length)
{
    unsigned long long i, j, k;
    rotor_init_genrand64(19650218ULL);
    i=1; j=0;
    k = (ROTOR_NN>key_length ? ROTOR_NN : key_length);
    for (; k; k--) {
        rotor_mt[i] = (rotor_mt[i] ^ ((rotor_mt[i-1] ^ (rotor_mt[i-1] >> 62)) * 3935559000370003845ULL))
          + init_key[j] + j; /* non linear */
        i++; j++;
        if (i>=ROTOR_NN) { rotor_mt[0] = rotor_mt[ROTOR_NN-1]; i=1; }
        if (j>=key_length) j=0;
    }
    for (k=ROTOR_NN-1; k; k--) {
        rotor_mt[i] = (rotor_mt[i] ^ ((rotor_mt[i-1] ^ (rotor_mt[i-1] >> 62)) * 2862933555777941757ULL))
          - i; /* non linear */
        i++;
        if (i>=ROTOR_NN) { rotor_mt[0] = rotor_mt[ROTOR_NN-1]; i=1; }
    }

    rotor_mt[0] = 1ULL << 63; /* MSB is 1; assuring non-zero initial array */ 
}
#endif

/* generates a random number on [0, 2^64-1]-interval */
unsigned long long rotor_genrand64_int64(void)
{
    int i;
    unsigned long long x;
    static unsigned long long mag01[2]={0ULL, ROTOR_MATRIX_A};

    if (rotor_mti >= ROTOR_NN) { /* generate ROTOR_NN words at one time */

        /* if rotor_init_genrand64() has not been called, */
        /* a default initial seed is used     */
        if (rotor_mti == ROTOR_NN+1) 
            rotor_init_genrand64(5489ULL); 

        for (i=0;i<ROTOR_NN-ROTOR_MM;i++) {
            x = (rotor_mt[i]&ROTOR_UM)|(rotor_mt[i+1]&ROTOR_LM);
            rotor_mt[i] = rotor_mt[i+ROTOR_MM] ^ (x>>1) ^ mag01[(int)(x&1ULL)];
        }
        for (;i<ROTOR_NN-1;i++) {
            x = (rotor_mt[i]&ROTOR_UM)|(rotor_mt[i+1]&ROTOR_LM);
            rotor_mt[i] = rotor_mt[i+(ROTOR_MM-ROTOR_NN)] ^ (x>>1) ^ mag01[(int)(x&1ULL)];
        }
        x = (rotor_mt[ROTOR_NN-1]&ROTOR_UM)|(rotor_mt[0]&ROTOR_LM);
        rotor_mt[ROTOR_NN-1] = rotor_mt[ROTOR_MM-1] ^ (x>>1) ^ mag01[(int)(x&1ULL)];

        rotor_mti = 0;
    }
  
    x = rotor_mt[rotor_mti++];

    x ^= (x >> 29) & 0x5555555555555555ULL;
    x ^= (x << 17) & 0x71D67FFFEDA60000ULL;
    x ^= (x << 37) & 0xFFF7EEE000000000ULL;
    x ^= (x >> 43);

    return x;
}

#if defined(_USED_)

/* generates a random number on [0, 2^63-1]-interval */
long long rotor_genrand64_int63(void)
{
    return (long long)(rotor_genrand64_int64() >> 1);
}

/* generates a random number on [0,1]-real-interval */
double rotor_genrand64_real1(void)
{
    return (rotor_genrand64_int64() >> 11) * (1.0/9007199254740991.0);
}

/* generates a random number on [0,1)-real-interval */
double rotor_genrand64_real2(void)
{
    return (rotor_genrand64_int64() >> 11) * (1.0/9007199254740992.0);
}

/* generates a random number on (0,1)-real-interval */
double rotor_genrand64_real3(void)
{
    return ((rotor_genrand64_int64() >> 12) + 0.5) * (1.0/4503599627370496.0);
}
#endif

